/***************************************************************************
 *
 * Copyright (C) 2001 International Business Machines
 * All rights reserved.
 *
 * This file is part of the GPFS mmfslinux kernel module.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer. 
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written
 *     permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *************************************************************************** */
/*
 * $Id: kx.c,v 1.22 2001/12/12 03:42:37 wsawdon Exp $
 *
 * $Log: kx.c,v $
 * Revision 1.22  2001/12/12 03:42:37  wsawdon
 * Changed trace statement -- no functional change.
 *
 * Revision 1.21  2001/09/29 02:00:35  manoj
 * Fix for Raleigh defect 6854: Ensure that stale pages in Linux's page cache
 * get invalidated on a BR token revoke in both the daemon and kernel versions
 * of flushMappedPages.
 *
 * Revision 1.20  2001/09/28 20:46:55  wyllie
 * Include more operations in vfsstats counters
 *
 * Revision 1.19  2001/09/25 18:19:50  gjertsen
 * Remove obsolete IA64 code.
 *
 * Revision 1.18  2001/09/12 05:45:51  schmuck
 * On ACL changes, permission information cached in the Linux inode
 * was not being updated correctly.
 *
 * Revision 1.17  2001/09/10 04:33:34  manoj
 * Fix for Raleigh defect 5687 and others. "Path walks" in Windows don't require
 * execute permission on directory components like they do on Unix. Access
 * for file lookup only requires FILE_READ_DATA permission on the parent
 * directory.  File operations require FILE_READ_DATA on parent and desired
 * access on target. Directory operations require desired access on target and
 * no parent directory permissions. Allow path_walk access for Windows callers
 * and use gpfsCheckNTAccess (through nss_ntaccess) for specific NT permissions.
 *
 * Revision 1.16  2001/09/05 23:40:52  jpalmer
 * Fix potential SMP problem using two writes to the oplock break pipe in breakSMBoplock
 *
 * Revision 1.15  2001/08/27 16:58:10  eshel
 * Defect 4515: complete file struct initialization with pointer to vfsmount.
 *
 * Revision 1.14  2001/08/24 21:43:39  manoj
 * Break shared oplocks (to none) asynchronously according to CIFS spec. Samba will
 * explicitly release kernel oplocks that are exclusively held.
 *
 * Revision 1.13  2001/08/23 22:55:29  manoj
 * Define an interface to allow a machine SID to be registered for access control.
 *
 * Revision 1.12  2001/08/16 21:22:45  manoj
 * Fix problems with notifying oplock breaks with wait queues. Pass openFile as an
 * argument to breakSMBOplock to check for oplock status.
 *
 * Revision 1.11  2001/08/10 18:22:31  gjertsen
 * Put errno predeclaration for kernel code in Shark-gpl.h.
 *
 * Revision 1.10  2001/08/02 12:15:40  jpalmer
 * Implement global lock clamp.  Fix integrity hold in clamp/unclamp functions.  Fix passing fromNode for BR lock move.
 *
 * Revision 1.9  2001/07/17 20:48:56  wsawdon
 * Fixed gpfs_stat & fstat. Linux defines struct stat64 in 2 places
 * with slightly different definitions in each.
 *
 * Revision 1.8  2001/07/11 03:56:56  manoj
 * Define gpfsCheckNTAccess to check if NT caller has access to a file through
 * individual NT permission bits (other that standard rwx which can be checked by
 * access() call).
 *
 * Revision 1.7  2001/07/10 12:06:26  jpalmer
 * Add function to allow SMB Open and FCNTL tokens to be moved from one system to
 * another in response to the NAS load balancer moving the users.  Add the
 * external interface to provide a lock clamping function that will block new
 * locks during the time NFS is recovering its FCNTL and File locks.
 *
 * Revision 1.6  2001/05/30 23:51:21  manoj
 * user_path_walk returns negative return codes on error.
 *
 * Revision 1.5  2001/05/25 14:48:20  gjertsen
 * Minor fixes to get IA64 code to compile again.
 *
 * Revision 1.4  2001/05/08 13:40:36  dixonbp
 * kxRegisterCleanup for linux and gpfs_f_cleanup/gpfsCleanup
 * to do the equivalent of what fopClose does on aix.
 *
 * Revision 1.3  2001/04/08 22:18:30  dcraft
 * Fix multinde delete race conditions.  Still incomplete.
 *
 * Revision 1.2  2001/04/05 18:16:59  schmuck
 * Fix compile errors in SMB_LOCKS code due to conversion to C.
 *
 * Revision 1.1  2001/03/30 21:36:15  dixonbp
 * Convert kx.C to kx.c
 *
 * Revision 1.30  2001/03/28 08:50:15  manoj
 * Support for DOS attributes (NSS only). Define gpfs_ops GetDosAttr and
 * SetDosAttr for the purpose.
 *
 * Revision 1.29  2001/03/22 03:50:45  schmuck
 * tsfattr: don't clobber tsfattrReasonCodeInfo after returning from gpfsFattr.
 *
 * Revision 1.28  2001/03/20 01:17:40  manoj
 * Added a new parameter to pass cred to gpfsGetAcl required to validate user
 * access for NT ACLs. Call setCred to get cred structure.
 *
 * Revision 1.27  2001/03/14 01:01:00  manoj
 * Run indent. Fix wait queues.
 *
 * Revision 1.26  2001/03/12 22:29:18  schmuck
 * Add hooks for oplock testing.
 *
 * Revision 1.25  2001/03/12 19:11:54  manoj
 * Add kernel oplock support directly into GPFS mmfslinux instead of being a
 * separate module.
 *
 * Revision 1.24  2001/03/11 22:04:43  dcraft
 * Must invalidate inode (permission) if ACL is updated/created
 *
 * Revision 1.23  2001/03/10 02:42:20  schmuck
 * Add timeout parameter for oplock break.
 *
 * Revision 1.22  2001/03/09 18:10:51  schmuck
 * More SMB oplock code.
 *
 * Revision 1.21  2001/03/09 02:34:05  schmuck
 * Tweak
 *
 * Revision 1.20  2001/03/09 02:28:28  schmuck
 * Changes for SMB oplocks.
 *
 * Revision 1.19  2001/02/08 22:08:39  eshel
 * if rCodeP was specified then set reason field
 *
 * Revision 1.18  2001/02/08 19:45:53  gjertsen
 * Add kxMadvise kernel extension
 *
 * Revision 1.17  2000/12/15 13:56:44  gjertsen
 * Clean up documentation.
 *
 * Revision 1.16  2000/12/01 02:10:59  schmuck
 * Instead of assigning NULL function pointers when initializing or resetting the
 * gpfs_operations table, have it point to a dummy function that returns ENOSYS.
 * This avoids having to check for NULL before each call.
 *
 * Revision 1.15  2000/11/16 01:01:04  wyllie
 * Split definition of ext_cred_t and the ACL helper routines that examine
 * it into platform-dependent versions, and simplify the Linux versions.
 * Give ACL helper routines cxi prefixes.
 *
 * Revision 1.14  2000/11/07 00:16:22  eshel
 * Add code to support remount.
 *
 * Revision 1.13  2000/11/06 19:56:14  gjertsen
 * Linux code cleanup and put in build safeguards.
 *
 * Revision 1.12  2000/11/03 20:27:01  dcraft
 * Build SMP, UP, NOHIGHMEM, and 4G memory variations of mmfslinux in
 * one pass.   Names are mmfslinux_UP1G, mmfslinux_UP4G, mmfslinux_SMP1G,
 * mmfslinux_SMP4G.
 *
 * Revision 1.11  2000/11/02 19:46:26  gjertsen
 * Linux code split. Pull out NBD stuff.
 *
 */

#include <Shark-gpl.h>

#ifdef MODULE
#include <linux/module.h>
#endif
#include <linux/string.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/dcache.h>
#include <linux/file.h>
#include <linux/stat.h>
#include <linux/fcntl.h>

/* grab stuff for madvise */
#define __KERNEL_SYSCALLS__
#include <linux/unistd.h>
#include <linux/mman.h>

#include <cxiSystem.h>
#include <cxi2gpfs.h>
#include <cxiCred.h>

#include <linux2gpfs.h>
#include <Trace.h>
#include <cxiMode.h>
#include <cxiTSFattr.h>
#include <cxiVFSStats.h>

#ifdef GPFS_ARCH_IA64
#define stat64 stat
#endif /* GPFS_ARCH_IA64 */


/* check if inode belongs to GPFS */
#define GPFS_TYPE(IP) (!(iP) ? false : \
                        (!(iP->i_sb) ? false : \
                          (!(iP->i_sb->s_type) ? false : \
                            !strcmp(iP->i_sb->s_type->name, "gpfs"))))

int
kxPoll(void* fdListP, int nFDs, int timeoutMS)
{
  return -ENOSYS;
}

int Cleanup_fd = 0;

/* In order to setup our termination callback routine (gpfs_f_cleanup)
   we create a dummy file and add it to our file table.  Then, upon
   process termination, the release file operation will be called in
   order to close the file.  The only operation we define for this
   dummy file is release (gpfs_f_cleanup). */
int
kxRegisterCleanup()
{
  int code = 0, rc = 0;
  struct inode *iP = NULL;
  struct file *fileP = NULL;
  struct dentry *dentryP = NULL;

  /* Make sure we only create one file */
  if (Cleanup_fd)
    return EEXIST;

  /* Allocate a file struct */
  fileP = get_empty_filp();
  if (!fileP)
  {
    code = 2;
    rc = ENFILE;
    goto xerror;
  }

  /* Allocate an inode struct */
  iP = get_empty_inode();
  if (!iP)
  {
    code = 3;
    rc = ENOMEM;
    goto xerror;
  }

  /* Allocate an available file descriptor */
  Cleanup_fd = get_unused_fd();
  if (Cleanup_fd < 0)
  {
    code = 4;
    rc = ENFILE;
    goto xerror;
  }


  /* Allocate a dentry sruct */
  dentryP = dget(d_alloc_root(iP));
  if (!dentryP)
  {
    code = 5;
    rc = ENOMEM;
    goto xerror;
  }

  /* Initialize and chain our file sructures */
  fileP->f_dentry = dentryP;
  fileP->f_op = &gpfs_cleanup_fops;
  fileP->f_flags = O_RDONLY;
  fileP->f_vfsmnt = current->fs->rootmnt;
  atomic_set(&fileP->f_count, 1);
  iP->i_mode = S_IFREG;

  /* Install the descriptor so it gets "closed" upon our termination */
  fd_install(Cleanup_fd, fileP);

  /* Once the descriptor for this dummy file is added to our file table,
     it is inherrited by all the processes of the daemon.  As each
     terminates, the files->count is decremented and on the last process
     termination all the descriptors will be closed by filp_close.

     The one catch here is that our file table is inherrited by the
     kernel threads we start as well as user processes.  This would
     cause a problem in that daemon termination does not include these
     kernel threads which aren't killed until restart (and therefore
     the file is never closed).  In order for our operation to be
     driven at daemon termiation, we must remove the file table from
     these kernel threads.  This is done in pagerKprocMainThreadP which
     begins our only such kernel threads. */

xerror:
  TRACE4(TRACE_VNODE, 1, TRCID_KXREGISTERCLEANUP_EXIT,
        "kxRegisterCleanup: fd %d iP %X rc %d code %d\n", Cleanup_fd, iP, rc, code);

  if (rc)
  {
    if (dentryP);
      dput(dentryP);
    if (Cleanup_fd)
      put_unused_fd(Cleanup_fd);
    if (iP)
      iput(iP);
    if (fileP)
      put_filp(fileP);

    Cleanup_fd = 0;
  }

  return rc;
}

void static
vstat(cxiVattr_t *vattrp, struct stat64 *statbuf)
{
  memset(statbuf, 0, sizeof(struct stat64));

  statbuf->st_dev         = vattrp->va_dev;
  statbuf->st_ino         = vattrp->va_serialno;
  statbuf->st_mode        = vattrp->va_mode;
  statbuf->st_nlink       = vattrp->va_nlink;
  statbuf->st_uid         = vattrp->va_uid;
  statbuf->st_gid         = vattrp->va_gid;
  statbuf->st_rdev        = vattrp->va_rdev;
  statbuf->st_size        = vattrp->va_size;
  statbuf->st_atime       = vattrp->va_atime.tv_sec;
  statbuf->st_mtime       = vattrp->va_mtime.tv_sec;
  statbuf->st_ctime       = vattrp->va_ctime.tv_sec;
  statbuf->st_blksize     = vattrp->va_blocksize;
  statbuf->st_blocks      = vattrp->va_blocks;

#ifdef STAT64_HAS_BROKEN_ST_INO
  /* Linux has 2 struct stat64 definitions:
      1) /usr/include/asm/stat.h
      2) /usr/include/bits/stat.h
     Of course, they differ
      1) st_dev & st_rdev is 2 bytes in (1) and 8 bytes in (2),
         but the 2 definitions overlap.
      2) st_ino is 8 bytes in (1) and 4 bytes in (2)
         and they are in different places!
    Fortunately, (1) defines an ifdef telling us to assign st_ino
    to a second variable which just happens to exactly match the
    definition in (2). */
  statbuf->__st_ino       = vattrp->va_serialno;
#endif
}

/* a call to return exact file status information
 * using the file's pathname;
 */
int
tsstat(char *pathname, struct stat64 *statP)
{
  struct gpfsVfsData_t *privVfsP;
  cxiNode_t *cnP;
  int code = 0;
  int rc = 0;
  struct nameidata nd;
  struct inode *iP;
  Boolean relePath = false;
  cxiVattr_t vattr;
  struct stat64 statbuf;

  if (pathname == NULL || statP == NULL)
  {
    code = 1;
    rc = EINVAL;
    goto xerror;
  }

  rc = user_path_walk(pathname, &nd);
  if (rc)
  {
    rc = -rc;
    code = 2;
    goto xerror;
  }
  relePath = true;

  iP = nd.dentry->d_inode;
  DBGASSERT(iP != NULL);

  TRACE2(TRACE_VNODE, 1, TRCID_LINUXOPS_TSSTAT_ENTER,
         "tsstat enter: iP 0x%lX statP 0x%lX\n", iP, statP);

  if (!GPFS_TYPE(iP))
  {
    code = 3;
    rc = EINVAL;
    goto xerror;
  }

  privVfsP = VP_TO_PVP(iP);
  if (privVfsP == NULL)
  {
    code = 4;
    rc = ENOENT;
    goto xerror;
  }
  cnP = VP_TO_CNP(iP);

  rc = gpfs_ops.gpfsGetattr(privVfsP, cnP, &vattr, true);

  if (rc == 0)
  {
    vstat(&vattr, &statbuf);

    rc = cxiCopyOut((char *)&statbuf, (char *)statP, sizeof(struct stat64));
    if (rc != 0)
    {
      code = 5;
      goto xerror;
    }
  }

xerror:
  if (relePath)
    path_release(&nd);

  TRACE2(TRACE_VNODE, 1, TRCID_LINUXOPS_TSSTAT_EXIT,
         "tsstat exit: code %d rc %d\n", code, rc);
  return -rc;
}

int
tsfstat(int fileDesc, struct stat64 *statP)
{
  struct gpfsVfsData_t *privVfsP;
  cxiNode_t *cnP;
  struct file *fP;
  struct inode *iP;
  int code = 0;
  int rc;
  Boolean releFile = false;
  cxiVattr_t vattr;
  struct stat64 statbuf;

  if (statP == NULL)
  {
    code = 1;
    rc = EINVAL;
    goto xerror;
  }

  fP = fget(fileDesc);
  if (!fP)
  {
    code = 2;
    rc = EBADF;
    goto xerror;
  }
  releFile = true;

  iP = fP->f_dentry->d_inode;
  DBGASSERT(iP != NULL);

  TRACE4(TRACE_VNODE, 1, TRCID_LINUXOPS_TSFSTAT_ENTER,
         "tsfstat enter: fd %d fP 0x%lX iP 0x%lX statP 0x%lX\n",
         fileDesc, fP, iP, statP);

  if (!GPFS_TYPE(iP))
  {
    code = 3;
    rc = EINVAL;
    goto xerror;
  }

  privVfsP = VP_TO_PVP(iP);
  if (privVfsP == NULL)
  {
    code = 4;
    rc = ENOENT;
    goto xerror;
  }
  cnP = VP_TO_CNP(iP);

  rc = gpfs_ops.gpfsGetattr(privVfsP, cnP, &vattr, true);

  if (rc == 0)
  {
    vstat(&vattr, &statbuf);

    rc = cxiCopyOut((char *)&statbuf, (char *)statP, sizeof(struct stat64));
    if (rc != 0)
    {
      code = 5;
      goto xerror;
    }
  }

xerror:
  if (releFile)
    fput(fP);

  TRACE2(TRACE_VNODE, 1, TRCID_LINUXOPS_TSFSTAT_EXIT,
         "tsfstat exit: code %d rc %d\n", code, rc);
  return -rc;
}

int
tsfattr(int fileDesc, int command, void *argP,
        tsfattrReasonCodeInfo *rCodeP)
{
  struct gpfsVfsData_t *privVfsP;
  cxiNode_t *cnP;
  struct MMFSVInfo *vinfoP;
  ext_cred_t eCred;
  struct file *fP;
  struct inode *iP;
  int reason = 0;
  int code = 0;
  int flags;
  int rc;
  Boolean releFile = false;

  VFS_STAT_START(tsfattrCall);

#ifdef SMB_LOCKS_TEST
  /* hack to test setSMBOplock via tsfattr call */
  if (command == 999)
  {
    extern int setSMBOplock(int fileDesc, int accessWant, int oplockWant,
                            void *breakArgP, int *oplockGrantedP);
    int args[2];
    int oplockGranted;
    rc = cxiCopyIn((char *)argP, (char *)args, sizeof(args));
    if (rc)
      return -rc;
    rc = setSMBOplock(fileDesc, args[0], args[1], (void *)0xdeadbeef, &oplockGranted);
    if (rc)
      return rc;
    return oplockGranted;
  }
#endif

  /* if rCodeP was specified then immediately reset reason field */
  if (rCodeP != NULL)
  {
    rc = cxiCopyOut((char*)&reason, (char*)&(rCodeP->reason), sizeof(reason));
    if (rc != 0)
    {
      code = 1;
      goto xerror;
    }
  }

  fP = fget(fileDesc);
  if (!fP)
  {
    code = 2;
    rc = EBADF;
    goto xerror;
  }
  releFile = true;

  vinfoP = (struct MMFSVInfo *)fP->private_data;

  iP = fP->f_dentry->d_inode;
  DBGASSERT(iP != NULL);

  flags = cxiOpenFlagsXlate(fP->f_flags);

  TRACE8(TRACE_VNODE, 1, TRCID_LINUXOPS_TSFATTR_ENTER,
         "tsfattr enter: fd %d fP 0x%lX f_flags 0x%X flags 0x%X "
         "iP 0x%lX command %d argP 0x%lX rCodeP 0x%lX\n",
         fileDesc, fP, fP->f_flags, flags, iP, 
         command, argP, rCodeP);

  if (!GPFS_TYPE(iP))
  {
    code = 3;
    rc = EINVAL;
    goto xerror;
  }

  privVfsP = VP_TO_PVP(iP);
  if (privVfsP == NULL)
  {
    code = 4;
    rc = ENOENT;
    goto xerror;
  }
  cnP = VP_TO_CNP(iP);

  setCred(&eCred);

  rc = gpfs_ops.gpfsFattr(privVfsP, cnP, vinfoP, flags, command, argP,
                          rCodeP, &eCred);

xerror:
  if (releFile)
    fput(fP);

  TRACE2(TRACE_VNODE, 1, TRCID_LINUXOPS_TSFATTR_EXIT,
         "tsfattr exit: code %d rc %d\n", code, rc);

  VFS_STAT_STOP;
  return -rc;
}

int
tsattr(char *pathname, int command, void *argP,
       tsfattrReasonCodeInfo *rCodeP)
{
  struct gpfsVfsData_t *privVfsP;
  cxiNode_t *cnP;
  int code = 0;
  int rc = 0;
  int reason = 0;       /* local reason code for resetting caller's */
  ext_cred_t eCred;
  struct nameidata nd;
  struct inode *iP;
  Boolean relePath = false;

  /* if rCodeP was specified then immediately reset reason field */
  if (rCodeP != NULL)
  {
    rc = cxiCopyOut((char*)&reason, (char*)&(rCodeP->reason), sizeof(reason));
    if (rc != 0)
    {
      code = 1;
      goto xerror;
    }
  }

  rc = user_path_walk(pathname, &nd);
  if (rc)
  {
    rc = -rc;
    code = 2;
    goto xerror;
  }
  relePath = true;

  iP = nd.dentry->d_inode;
  DBGASSERT(iP != NULL);

  TRACE4(TRACE_VNODE, 1, TRCID_LINUXOPS_TSATTR_ENTER,
         "tsattr enter: iP 0x%lX command %d argP 0x%lX rCodeP 0x%lX\n",
         iP, command, argP, rCodeP);

  if (!GPFS_TYPE(iP))
  {
    code = 3;
    rc = EINVAL;
    goto xerror;
  }

  privVfsP = VP_TO_PVP(iP);
  if (privVfsP == NULL)
  {
    code = 4;
    rc = ENOENT;
    goto xerror;
  }
  cnP = VP_TO_CNP(iP);

  setCred(&eCred);
  rc = gpfs_ops.gpfsFattr(privVfsP, cnP, NULL, 0, command, argP, rCodeP,
                          &eCred);

xerror:
  if (relePath)
    path_release(&nd);

  TRACE2(TRACE_VNODE, 1, TRCID_LINUXOPS_TSATTR_EXIT,
         "tsattr exit: code %d rc %d\n", code, rc);

  return -rc;
}

int
kxGetACL(char *pathname, int flags, void *aclP)
{
  int rc = 0;
  int code = 0;
  cxiNode_t *cnP;
  struct gpfsVfsData_t *privVfsP;
  ext_cred_t eCred;
  struct nameidata nd;
  struct inode *iP;
  Boolean relePath = false;

  /* Verify input exists */
  if (pathname == NULL || aclP == NULL)
  {
    code = 1;
    rc = EINVAL;
    goto xerror;
  }

  rc = user_path_walk(pathname, &nd);
  if (rc)
  {
    rc = -rc;
    code = 2;
    goto xerror;
  }
  relePath = true;

  iP = nd.dentry->d_inode;
  DBGASSERT(iP != NULL);

  if (!GPFS_TYPE(iP))
  {
    code = 3;
    rc = EINVAL;
    goto xerror;
  }

  privVfsP = VP_TO_PVP(iP);
  if (privVfsP == NULL)
  {
    code = 4;
    rc = ENOENT;
    goto xerror;
  }

  cnP = VP_TO_CNP(iP);

  /* Retrieve the ACL data */
  setCred(&eCred);
  rc = gpfs_ops.gpfsGetAcl(privVfsP, cnP, flags, aclP, &eCred);

xerror:
  if (relePath)
    path_release(&nd);

  TRACE2(TRACE_VNODE, 1, TRCID_LINUXOPS_KXGETACL_EXIT,
         "kxGetACL exit: code %d rc %d\n", code, rc);
  return -rc;
}

int
kxPutACL(char *pathname, int flags, void *aclP)
{
  int rc = 0;
  int code = 0;
  cxiNode_t *cnP;
  struct gpfsVfsData_t *privVfsP;
  ext_cred_t eCred;
  struct nameidata nd;
  struct inode *iP;
  Boolean relePath = false;

  /* Verify input exists */
  if (pathname == NULL || aclP == NULL)
  {
    code = 1;
    rc = EINVAL;
    goto xerror;
  }

  rc = user_path_walk(pathname, &nd);
  if (rc)
  {
    rc = -rc;
    code = 2;
    goto xerror;
  }
  relePath = true;

  iP = nd.dentry->d_inode;
  DBGASSERT(iP != NULL);

  if (!GPFS_TYPE(iP))
  {
    code = 3;
    rc = EINVAL;
    goto xerror;
  }

  privVfsP = VP_TO_PVP(iP);
  if (privVfsP == NULL)
  {
    code = 4;
    rc = ENOENT;
    goto xerror;
  }

  cnP = VP_TO_CNP(iP);

  /* Put the ACL data */
  setCred(&eCred);
  rc = gpfs_ops.gpfsPutAcl(privVfsP, cnP, flags, aclP, &eCred);

xerror:
  if (relePath)
    path_release(&nd);

  TRACE2(TRACE_VNODE, 1, TRCID_LINUXOPS_KXPUTACL_EXIT,
         "kxPutACL exit: code %d rc %d\n", code, rc);
  return -rc;
}

#ifdef CONFIG_NSS
int
kxGetDosAttr(char *pathname, int flags, unsigned char *attrP)
{
  int rc = 0;
  int code = 0;
  cxiNode_t *cnP;
  struct gpfsVfsData_t *privVfsP;
  ext_cred_t eCred;
  struct nameidata nd;
  struct inode *iP;
  Boolean relePath = false;

  /* Verify input exists */
  if (pathname == NULL || attrP == NULL)
  {
    code = 1;
    rc = EINVAL;
    goto xerror;
  }

  rc = user_path_walk(pathname, &nd);
  if (rc)
  {
    rc = -rc;
    code = 2;
    goto xerror;
  }
  relePath = true;

  iP = nd.dentry->d_inode;
  DBGASSERT(iP != NULL);

  if (!GPFS_TYPE(iP))
  {
    code = 3;
    rc = EINVAL;
    goto xerror;
  }

  privVfsP = VP_TO_PVP(iP);
  if (privVfsP == NULL)
  {
    code = 4;
    rc = ENOENT;
    goto xerror;
  }

  cnP = VP_TO_CNP(iP);

  /* Retrieve the ACL data */
  setCred(&eCred);
  rc = gpfs_ops.gpfsGetDosAttr(privVfsP, cnP, flags, attrP, &eCred);

xerror:
  if (relePath)
    path_release(&nd);

  TRACE2(TRACE_NSS, 1, TRCID_LINUXOPS_KXGETDOSATTR_EXIT,
         "kxGetDosAttr exit: code %d rc %d\n", code, rc);
  return -rc;
}

int
kxPutDosAttr(char *pathname, int flags, unsigned char attrP)
{
  int rc = 0;
  int code = 0;
  cxiNode_t *cnP;
  struct gpfsVfsData_t *privVfsP;
  ext_cred_t eCred;
  struct nameidata nd;
  struct inode *iP;
  Boolean relePath = false;

  /* Verify input exists */
  if (pathname == NULL)
  {
    code = 1;
    rc = EINVAL;
    goto xerror;
  }

  rc = user_path_walk(pathname, &nd);
  if (rc)
  {
    rc = -rc;
    code = 2;
    goto xerror;
  }
  relePath = true;

  iP = nd.dentry->d_inode;
  DBGASSERT(iP != NULL);

  if (!GPFS_TYPE(iP))
  {
    code = 3;
    rc = EINVAL;
    goto xerror;
  }

  privVfsP = VP_TO_PVP(iP);
  if (privVfsP == NULL)
  {
    code = 4;
    rc = ENOENT;
    goto xerror;
  }

  cnP = VP_TO_CNP(iP);

  /* Put the ACL data */
  setCred(&eCred);
  rc = gpfs_ops.gpfsPutDosAttr(privVfsP, cnP, flags, attrP, &eCred);

xerror:
  if (relePath)
    path_release(&nd);

  TRACE2(TRACE_NSS, 1, TRCID_LINUXOPS_KXPUTDOSATTR_EXIT,
         "kxPutDosAttr exit: code %d rc %d\n", code, rc);

  return -rc;
}

int
kxCheckNTAccess(char *pathname, UInt32 ntmask)
{
  int rc = 0;
  int code = 0;
  struct gpfsVfsData_t *privVfsP;
  ext_cred_t eCred;
  struct nameidata nd;
  struct inode *piP, *iP;
  Boolean relePath = false;

  /* Verify input exists */
  if (pathname == NULL)
  {
    code = 1;
    rc = EINVAL;
    goto xerror;
  }

  rc = user_path_walk(pathname, &nd);
  if (rc)
  {
  	rc = -rc;
	code = 2;
	goto xerror;
  }
  relePath = true;

  iP = nd.dentry->d_inode;
  DBGASSERT(iP != NULL);
  if (!GPFS_TYPE(iP))
  {
  	code = 3;
	rc = EINVAL;
	goto xerror;
  }
  privVfsP = VP_TO_PVP(iP);
  if (privVfsP == NULL)
  {
    code = 4;
	rc = ENOENT;
	goto xerror;
  }
  piP = nd.dentry->d_parent->d_inode;
  setCred(&eCred);
  rc = gpfs_ops.gpfsCheckNTAccess(privVfsP, VP_TO_CNP(piP), VP_TO_CNP(iP), 
                                  ntmask, &eCred);

xerror:
  if (relePath)
  	path_release(&nd);

  TRACE4(TRACE_NSS, 1, TRCID_LINUXOPS_KXNTACCESS_EXIT,
         "kxCheckNTAccess exit: pathname %s, mask 0x%x, code %d rc %d\n", 
         pathname, ntmask, code, rc);
  return -rc;
}

int kxSetMachineSID(void *sid)
{
  ext_cred_t eCred;
  int rc;

  setCred(&eCred);
  rc = gpfs_ops.gpfsSetMachineSID(sid, &eCred);
  return -rc;
}
#endif // CONFIG_NSS

/* madvise is handled as a kernel extension / ioctl call until support
   is provided by glibc */
static inline _syscall3(int,madvise,void*,addr,size_t,len,int,advice);

int
kxMadvise(void* addr, size_t len, int advice)
{
  return madvise(addr, len, advice);
}


#ifdef SMB_LOCKS
#include <oplock.h>
#include <asm/uaccess.h>
int
setSMBOplock(int fileDesc, int accessWant, int oplockWant, void *breakArgP,
             int *oplockGrantedP)
{
  struct gpfsVfsData_t *privVfsP;
  cxiNode_t *cnP;
  struct MMFSVInfo *vinfoP;
  struct file *fP;
  struct inode *iP;
  int reason = 0;
  int code = 0;
  int flags;
  int rc;
  Boolean releFile = false;

  /* get file pointer and inode */
  fP = fget(fileDesc);
  if (!fP)
  {
    code = 2;
    rc = EBADF;
    goto xerror;
  }
  releFile = true;

  vinfoP = (struct MMFSVInfo *)fP->private_data;

  iP = fP->f_dentry->d_inode;
  DBGASSERT(iP != NULL);

  TRACE5(TRACE_VNODE, 1, TRCID_LINUXOPS_SETSMBOPLOCK_E,
         "setSMBOplock enter: fd %d fP 0x%lX acc %d oplock %d argP 0x%lX",
         fileDesc, fP, accessWant, oplockWant, breakArgP);

  if (!GPFS_TYPE(iP))
  {
    code = 3;
    rc = EINVAL;
    goto xerror;
  }

  privVfsP = VP_TO_PVP(iP);
  if (privVfsP == NULL)
  {
    code = 4;
    rc = ENOENT;
    goto xerror;
  }
  cnP = VP_TO_CNP(iP);

  rc = gpfs_ops.gpfsSetSMBOplock(privVfsP, cnP, vinfoP, accessWant,
                                 oplockWant, breakArgP, oplockGrantedP);

xerror:
  if (releFile)
    fput(fP);

  TRACE2(TRACE_VNODE, 1, TRCID_LINUXOPS_SETSMBOPLOCK_X,
         "setSMBOplock exit: granted %d rc %d", *oplockGrantedP, rc);

  return -rc;
}

int
breakSMBOplock(cxiDev_t dev, cxiIno_t ino, int oplockCurrent,
               void *breakArgP, void *fileArgP, int timeoutSeconds)
{
  DECLARE_WAITQUEUE(wait, current);
  extern wait_queue_head_t oplock_queue;
  struct inode *inode;
  char dummy;

  /* The slightly unusual definition of the following structure is to
   * insure that there are no pad bytes in the middle. */
  oplock_stat_t os;
  struct  {
    char    dummy_byte;
    char    os_space[sizeof os];}
            writeBlock;
  oplock_stat_t* osP = (oplock_stat_t *)&writeBlock.os_space;
  struct file *pfilp = (struct file *)breakArgP;
  int err;
  signed long timeout;

  mm_segment_t oldfs;
  unsigned long flags;
  sigset_t oldset;

#ifdef SMB_LOCKS_TEST
  /* handle breaking of fake oplocks for testing purposes */
  if (breakArgP == (void *)0xdeadbeef)
    return 0;
#endif

#if 0
  /* Samba wants ulong for ino_t and ulong long int for dev_t */
  inode = dentry->d_inode;
  if (!inode) {
    /*
     * This seems to be an oplock break for a temporary file used
     * by NFS. XXXX Don't know what to do here - return success
     */
    return 0;
  }
#endif

  if (!pfilp || !pfilp->f_op || !pfilp->f_op->write) {
    err = EINVAL;
    goto xerror;
  }

  writeBlock.dummy_byte = 0;
  osP->os_dev = dev;
  osP->os_ino = ino;

  oldfs = get_fs();
  set_fs(get_ds());

  spin_lock_irqsave(&current->sigmask_lock, flags);
  oldset = current->blocked;
  sigfillset(&current->blocked);
  recalc_sigpending(current);
  spin_unlock_irqrestore(&current->sigmask_lock, flags);

  /*
   * Write one byte of zero to send a kernel break notify
   * message followed by the status structure
   */
  err = pfilp->f_op->write(pfilp, (char *)&writeBlock, sizeof(writeBlock), &pfilp->f_pos);
  if (err < 0)
    printk("breakSMBoplock: write on pipe failed; err %d\n", err);
  else {
    err = 0;
  }

  spin_lock_irqsave(&current->sigmask_lock, flags);
  current->blocked = oldset;
  recalc_sigpending(current);
  spin_unlock_irqrestore(&current->sigmask_lock, flags);
  set_fs(oldfs);

  /*
   * Setup timer and schedule here to wait for oplock
   * break to be serviced before returning.
   * We don't really need this for most applications; we could
   * have just dropped the packet (and returned EACCES) and
   * the client would keep trying until samba forced oplock
   * break. This is cleaner, though
   * We don't want to schedule forever if the samba process that will
   * deliver the wakeup call has terminated - only sleep for timeoutSeconds.
   */
		
#if 1
  if (oplockCurrent == smbOplockExclusive)
  {
    add_wait_queue(&oplock_queue, &wait);
    timeout = timeoutSeconds * HZ;
    while (timeout > 0) {
      set_current_state(TASK_INTERRUPTIBLE);
      /* Check whether the oplock has been released or downgraded */
      if (gpfs_ops.SMBGetOplockState(fileArgP) < oplockCurrent)
        break;
      timeout = schedule_timeout(timeout);
    }
    set_current_state(TASK_RUNNING);
    remove_wait_queue(&oplock_queue, &wait);
  }
#if 0
  if (current->signal & ~current->blocked) /* a signal arrived */
    return -ERESTARTSYS;
#endif
#endif

 xerror:
  return err;
}


int
kxClampLocks(unsigned long argSB)
{
  int rc;
  struct super_block* sbP = (struct super_block*) argSB;
  struct gpfsVfsData_t *privVfsP;

  TRACE2(TRACE_SMB, 5, TRCID_KXCLAMPLOCKS,
         "kxClampLocks: SuperBlock: 0x%X, gpfsClampLocks: 0x%X",
          sbP, gpfs_ops.gpfsClampLocks);

  if (!cxiIsSuperUser())
    return -1;

  if (sbP==NULL)
    privVfsP = NULL;
  else
    privVfsP = (struct gpfsVfsData_t*)sbP->u.generic_sbp;

  rc = gpfs_ops.gpfsClampLocks(privVfsP, sbP, true);

  return rc;
}


int
kxUnClampLocks(unsigned long argSB)
{
  int rc;
  struct super_block* sbP = (struct super_block*) argSB;
  struct gpfsVfsData_t *privVfsP;

  if (!cxiIsSuperUser())
    return -1;

  if (sbP==NULL)
    privVfsP = NULL;
  else
    privVfsP = (struct gpfsVfsData_t*)sbP->u.generic_sbp;

  rc = gpfs_ops.gpfsClampLocks(privVfsP, sbP, false);

  return rc;
}
#endif
